<?php

/*************************************************
## encrypt_secret
*************************************************/
function grogin_cipher_key() {
	$base = (defined('SECURE_AUTH_KEY') ? SECURE_AUTH_KEY : '') .
	        (defined('SECURE_AUTH_SALT') ? SECURE_AUTH_SALT : '');
	if ($base === '') $base = 'grogin-fallback-change-me';
	return hash('sha256', $base, true); // 32 bytes
}

function grogin_encrypt($plain) {
	if ($plain === '' || $plain === null) return '';
	if (!function_exists('openssl_encrypt')) return base64_encode($plain); // fallback
	$iv  = random_bytes(16);
	$ct  = openssl_encrypt($plain, 'AES-256-CBC', grogin_cipher_key(), OPENSSL_RAW_DATA, $iv);
	return $ct === false ? '' : base64_encode($iv.$ct);
}

function grogin_decrypt($enc) {
	if ($enc === '' || $enc === null) return '';
	if (!function_exists('openssl_decrypt')) return base64_decode($enc); // fallback
	$raw = base64_decode($enc, true);
	if (!$raw || strlen($raw) < 17) return '';
	$iv = substr($raw, 0, 16);
	$ct = substr($raw, 16);
	$pt = openssl_decrypt($ct, 'AES-256-CBC', grogin_cipher_key(), OPENSSL_RAW_DATA, $iv);
	return $pt === false ? '' : $pt;
}

function grogin_kirki_sanitize_encrypt_secret($value) {
	if (empty($value)) {
		return get_option('grogin_openai_secret_key_enc', '');
	}

	$enc = grogin_encrypt(trim($value));
	if ($enc === '') {
		return get_option('grogin_openai_secret_key_enc', '');
	}

	add_option('grogin_openai_secret_key_enc', $enc, '', 'no');
	update_option('grogin_openai_secret_key_enc', $enc, false);
	global $wpdb;
	$wpdb->query( $wpdb->prepare(
		"UPDATE {$wpdb->options} SET autoload='no' WHERE option_name=%s",
		'grogin_openai_secret_key_enc'
	) );

	return $enc;
}

/*************************************************
##Define OpenAI key
*************************************************/
function grogin_get_openai_api_key() {
	if ( !( is_admin() || (defined('DOING_AJAX') && DOING_AJAX) ) ) {
		return '';
	}
	$enc = get_option('grogin_openai_secret_key_enc', '');
	return grogin_decrypt($enc);
}

if (!defined('KLB_OPENAI_API_KEY')) {
	define('KLB_OPENAI_API_KEY', grogin_get_openai_api_key());
}

/*************************************************
## Helper – OpenAI Responses API
*************************************************/
function klb_ai_call_openai_responses( $prompt, $model = 'gpt-4.1-mini', $timeout = 35 ) {
	if ( ! defined('KLB_OPENAI_API_KEY') || KLB_OPENAI_API_KEY === '' ) {
		return new WP_Error('klb_ai_no_key', esc_html__('OpenAI API key not defined.', 'grogin-core'));
	}
	$endpoint = 'https://api.openai.com/v1/responses';
	$body = [
		'model' => $model,
		'input' => $prompt,
	];
	$res = wp_remote_post($endpoint, [
		'timeout' => $timeout,
		'headers' => [
			'Authorization' => 'Bearer ' . KLB_OPENAI_API_KEY,
			'Content-Type'  => 'application/json',
		],
		'body' => wp_json_encode($body),
	]);

	if ( is_wp_error($res) ) return $res;
	$code = wp_remote_retrieve_response_code($res);
	$raw  = wp_remote_retrieve_body($res);
	if ( $code < 200 || $code >= 300 ) {
		return new WP_Error('klb_ai_http', esc_html__('OpenAI API error:', 'grogin-core') . ' ' . $raw);
	}

	$data = json_decode($raw, true);
	if ( ! is_array($data) ) {
		return new WP_Error('klb_ai_json', esc_html__('Invalid JSON response from OpenAI.', 'grogin-core'));
	}

	$text = '';
	if ( isset($data['output_text']) && is_string($data['output_text']) ) {
		$text = trim($data['output_text']);
	} elseif ( isset($data['output'][0]['content'][0]['text']) ) {
		$text = trim($data['output'][0]['content'][0]['text']);
	}

	if ( $text === '' ) {
		return new WP_Error('klb_ai_empty', esc_html__('Empty or unexpected model output.', 'grogin-core'));
	}

	return $text;
}

/** 3) Add Meta Box with Buttons */
add_action('add_meta_boxes', function(){
	add_meta_box(
		'klb-ai-desc',
		esc_html__('KLB AI', 'grogin-core'),
		function($post){
			wp_nonce_field('klb_ai_desc_nonce', 'klb_ai_desc_nonce');
			echo '<p>' . esc_html__('Use AI to generate product content from the title.', 'grogin-core') . '</p>';
			echo '<button type="button" class="klb-ai-button  button" id="klb-ai-generate-full" data-post="' . esc_attr($post->ID) . '">' . esc_html__('Generate Full Description (AI)', 'grogin-core') . '</button><br><br>';
			echo '<button type="button" class="klb-ai-button button button-primary" id="klb-ai-generate-short" data-post="' . esc_attr($post->ID) . '">' . esc_html__('Generate Short Description (AI)', 'grogin-core') . '</button>';
			echo '<div id="klb-ai-status" style="margin-top:10px;"></div>';
		},
		'product', 'side', 'high'
	);
});

/** 4) JS for Buttons (live update) */
add_action('admin_footer', function () {
	global $typenow;
	if ( $typenow !== 'product' ) return;

	$ajax  = admin_url('admin-ajax.php');
	$nonce = wp_create_nonce('klb_ai_desc_nonce');
	?>
	<style>
		.klb-ai-button {
			background: #000 !important;
			width: 100%;
			color: #fff !important;
			border-color: #000 !important;
		}
	</style>
	<script type="text/javascript">
	function klbUpdateExcerpt(text){
		if(!text) return;
		var $ = jQuery, $el = $('#excerpt');
		if(!$el.length){ $el = $('textarea[data-id="excerpt"]'); }
		if($el.length){ $el.val(text).trigger('input').trigger('change'); }
		if (typeof tinymce !== 'undefined'){ var ed = tinymce.get('excerpt'); if (ed) { ed.setContent(text); ed.save(); } }
		if (window.wp && wp.data && wp.data.dispatch){
			try { wp.data.dispatch('core/editor').editPost({ excerpt: text }); } catch(e){}
			try { wp.data.dispatch('woocommerce/product').editProduct({ short_description: text }); } catch(e){}
			try { wp.data.dispatch('woocommerce/product-editor').editProduct({ short_description: text }); } catch(e){}
		}
	}
	function klbUpdateContent(text){
		if(!text) return;
		var $ = jQuery, $el = $('#content');
		if(!$el.length){ $el = $('textarea[data-id="content"]'); }
		if($el.length){ $el.val(text).trigger('input').trigger('change'); }
		if (typeof tinymce !== 'undefined'){ var ed = tinymce.get('content'); if (ed) { ed.setContent(text); ed.save(); } }
		if (window.wp && wp.data && wp.data.dispatch){
			try { wp.data.dispatch('core/editor').editPost({ content: text }); } catch(e){}
			try { wp.data.dispatch('woocommerce/product').editProduct({ description: text }); } catch(e){}
			try { wp.data.dispatch('woocommerce/product-editor').editProduct({ description: text }); } catch(e){}
		}
	}

	(function($){
		function klbCallAI(actionType, btn){
			var $btn = $(btn), $st = $('#klb-ai-status');
			var postID = $btn.data('post');
			var action = (actionType === 'full') ? 'klb_ai_generate_full' : 'klb_ai_generate_short';
			$btn.prop('disabled', true);
			$st.text(actionType === 'full' ? '<?php echo esc_js( __( "Generating full description…", "grogin-core" ) ); ?>' : '<?php echo esc_js( __( "Generating short description…", "grogin-core" ) ); ?>');

			$.post('<?php echo esc_js($ajax); ?>', {
				action: action,
				post_id: postID,
				nonce:  '<?php echo esc_js($nonce); ?>'
			})
			.done(function(res){
				if(res && res.success){
					$st.text(res.data && res.data.message ? res.data.message : '<?php echo esc_js( __( "Done.", "grogin-core" ) ); ?>');
					if(res.data && res.data.text){
						if(actionType === 'full'){ klbUpdateContent(res.data.text); }
						else { klbUpdateExcerpt(res.data.text); }
					}
				}else{
					$st.text((res && res.data && res.data.message) ? res.data.message : '<?php echo esc_js( __( "Error.", "grogin-core" ) ); ?>');
					console.error('KLB AI Error:', res);
				}
			})
			.fail(function(xhr){
				$st.text('<?php echo esc_js( __( "Request failed.", "grogin-core" ) ); ?> ' + (xhr && xhr.status ? xhr.status : ''));
				console.error('KLB AI Network Error:', xhr);
			})
			.always(function(){ $btn.prop('disabled', false); });
		}

		$(document).on('click', '#klb-ai-generate-short', function(){ klbCallAI('short', this); });
		$(document).on('click', '#klb-ai-generate-full', function(){ klbCallAI('full', this); });
	})(jQuery);
	</script>
	<?php
});

/** 5) AJAX: Short Description */
add_action('wp_ajax_klb_ai_generate_short', function(){
	if ( ! current_user_can('edit_products') ) wp_send_json_error(['message' => esc_html__('You are not authorized.', 'grogin-core')]);
	if ( empty($_POST['nonce']) || ! wp_verify_nonce($_POST['nonce'], 'klb_ai_desc_nonce') )
		wp_send_json_error(['message' => esc_html__('Invalid request (nonce).', 'grogin-core')]);

	$post_id = intval($_POST['post_id']);
	$title = get_the_title($post_id);
	if ( ! $title ) wp_send_json_error(['message' => esc_html__('Product title not found.', 'grogin-core')]);
	$length = get_theme_mod( 'grogin_product_short_description_length', array( 'min' => '60', 'max' => '90') );

	$prompt = "Generate a short product description based on this title.\n" .
		"Language: ".get_theme_mod('grogin_product_short_description_language').".\nLength: about ".$length['min']."–".$length['max']." words. The response must not be shorter than ".$length['min']." words.\n" .
		"Style: concise, informative, neutral, 2 short paragraphs.\n" .
		"Use the main keyword naturally in the first sentence.\n" .
		"Product title: {$title}";

	$res = klb_ai_call_openai_responses($prompt);
	if ( is_wp_error($res) ) wp_send_json_error(['message' => $res->get_error_message()]);

	wp_update_post(['ID' => $post_id, 'post_excerpt' => $res]);

	wp_send_json_success(['message' => esc_html__('Short description generated and saved.', 'grogin-core'), 'text' => $res]);
});

/** 6) AJAX: Full Description */
add_action('wp_ajax_klb_ai_generate_full', function(){
	if ( ! current_user_can('edit_products') ) wp_send_json_error(['message' => esc_html__('You are not authorized.', 'grogin-core')]);
	if ( empty($_POST['nonce']) || ! wp_verify_nonce($_POST['nonce'], 'klb_ai_desc_nonce') )
		wp_send_json_error(['message' => esc_html__('Invalid request (nonce).', 'grogin-core')]);

	$post_id = intval($_POST['post_id']);
	$title = get_the_title($post_id);
	if ( ! $title ) wp_send_json_error(['message' => esc_html__('Product title not found.', 'grogin-core')]);
	$length = get_theme_mod( 'grogin_product_description_length', array( 'min' => '150', 'max' => '200') );
	
	$prompt = "Generate a full product description based on this title.\n" .
		"Language: ".get_theme_mod('grogin_product_description_language').".\nLength: about ".$length['min']."–".$length['max']." words. The response must not be shorter than ".$length['min']." words.\n" .
		"Format: 3 paragraphs describing features, usage, and benefits.\n" .
		"Keep it natural, SEO-friendly, and avoid exaggeration.\n" .
		"Product title: {$title}";

	$res = klb_ai_call_openai_responses($prompt);
	if ( is_wp_error($res) ) wp_send_json_error(['message' => $res->get_error_message()]);

	wp_update_post(['ID' => $post_id, 'post_content' => $res]);

	wp_send_json_success(['message' => esc_html__('Full description generated and saved.', 'grogin-core'), 'text' => $res]);
});